-- ZHZ PowerSearch Database Migrations

-- Table: zhz_powersearch_sync_log
-- Stores product sync events
CREATE TABLE IF NOT EXISTS `zhz_powersearch_sync_log` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `product_id` INT NOT NULL,
    `event_type` VARCHAR(50) NOT NULL COMMENT 'created, updated, deleted',
    `status` VARCHAR(20) DEFAULT 'pending' COMMENT 'pending, success, failed',
    `error_message` TEXT NULL,
    `webhook_response` TEXT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_product_id` (`product_id`),
    INDEX `idx_event_type` (`event_type`),
    INDEX `idx_status` (`status`),
    INDEX `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: zhz_powersearch_search_cache
-- Caches search results
CREATE TABLE IF NOT EXISTS `zhz_powersearch_search_cache` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `search_hash` VARCHAR(64) NOT NULL UNIQUE COMMENT 'SHA256 hash of search query',
    `search_query` VARCHAR(500) NOT NULL,
    `cache_data` LONGTEXT NOT NULL COMMENT 'JSON encoded search results',
    `result_count` INT DEFAULT 0,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `expires_at` TIMESTAMP NULL,
    INDEX `idx_search_hash` (`search_hash`),
    INDEX `idx_expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: zhz_powersearch_sync_status
-- Tracks overall sync status
CREATE TABLE IF NOT EXISTS `zhz_powersearch_sync_status` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `total_products` INT DEFAULT 0,
    `synced_products` INT DEFAULT 0,
    `failed_products` INT DEFAULT 0,
    `last_sync_at` TIMESTAMP NULL,
    `next_sync_at` TIMESTAMP NULL,
    `sync_status` VARCHAR(20) DEFAULT 'idle' COMMENT 'idle, syncing, completed',
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert initial sync status
INSERT IGNORE INTO `zhz_powersearch_sync_status` (id, sync_status) VALUES (1, 'idle');

-- Table: zhz_powersearch_category_priority
-- Category priority configuration for product ranking
CREATE TABLE IF NOT EXISTS `zhz_powersearch_category_priority` (
    `kKategorie` INT PRIMARY KEY COMMENT 'Category ID from tkategorie',
    `priority` INT NOT NULL DEFAULT 50 COMMENT '0-100: Priority for search ranking',
    `featured` TINYINT(1) NOT NULL DEFAULT 0 COMMENT 'Featured categories appear first',
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_priority` (`priority`),
    INDEX `idx_featured` (`featured`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Category priority configuration for eazyfind.me ranking';

-- Set default priorities for common category patterns (safe - only runs if table is empty)
INSERT IGNORE INTO `zhz_powersearch_category_priority` (kKategorie, priority, featured)
SELECT
    k.kKategorie,
    CASE
        -- High priority (100) for Sale/Action categories
        WHEN k.cName LIKE '%Sale%' OR
             k.cName LIKE '%Aktion%' OR
             k.cName LIKE '%Angebot%' OR
             k.cName LIKE '%SALE%' OR
             k.cName LIKE '%Rabatt%' OR
             k.cName LIKE '%Discount%' THEN 100

        -- Medium-high priority (80) for New/Novelty categories
        WHEN k.cName LIKE '%Neu%' OR
             k.cName LIKE '%New%' OR
             k.cName LIKE '%Neuheit%' OR
             k.cName LIKE '%Nouveauté%' THEN 80

        -- Medium priority (60) for Bestseller categories
        WHEN k.cName LIKE '%Bestseller%' OR
             k.cName LIKE '%Empfohlen%' OR
             k.cName LIKE '%Featured%' OR
             k.cName LIKE '%Meistverkauft%' THEN 60

        -- Low priority (40) for Outlet/Clearance
        WHEN k.cName LIKE '%Restposten%' OR
             k.cName LIKE '%Outlet%' OR
             k.cName LIKE '%Clearance%' THEN 40

        -- Default priority (50)
        ELSE 50
    END as priority,
    CASE
        -- Featured flag for important categories
        WHEN k.cName LIKE '%Sale%' OR
             k.cName LIKE '%Aktion%' OR
             k.cName LIKE '%Neu%' OR
             k.cName LIKE '%New%' OR
             k.cName LIKE '%SALE%' THEN 1
        ELSE 0
    END as featured
FROM tkategorie k
WHERE k.kKategorie NOT IN (SELECT kKategorie FROM zhz_powersearch_category_priority);
